<?php
// wcf imports
require_once(WCF_DIR.'lib/data/user/gallery/UserGalleryPhoto.class.php');
require_once(WCF_DIR.'lib/data/image/Thumbnail.class.php');

/**
 * Provides functions to manage gallery photos.
 *
 * @author 	Marcel Werk
 * @copyright	2001-2009 WoltLab GmbH
 * @license	WoltLab Burning Board License <http://www.woltlab.com/products/burning_board/license.php>
 * @package	com.woltlab.wcf.user.gallery
 * @subpackage	data.user.gallery
 * @category 	Community Framework (commercial)
 */
class UserGalleryPhotoEditor extends UserGalleryPhoto {
	private static $allowedFileExtensions = null;
	private static $illegalFileExtensions = array('php', 'php3', 'php4', 'php5', 'phtml');

	/**
	 * Creates a new gallery photo.
	 * 
	 * @param	integer		$ownerID
	 * @param	string		$tmpName
	 * @param	string		$filename
	 * @param	string		$title
	 * @param	string		$description
	 * @param	integer		$albumID
	 * @return	UserGalleryPhotoEditor
	 */
	public static function create($ownerID, $tmpName, $filename, $title, $description = '', $albumID = 0) {
		// check image content
		if (!ImageUtil::checkImageContent($tmpName)) {
			throw new UserInputException('upload', 'badImage');
		}
		
		// get file extension
		/*$fileExtension = '';
		if (!empty($filename) && StringUtil::indexOf($filename, '.') !== false) {
			$fileExtension = StringUtil::toLowerCase(StringUtil::substring($filename, StringUtil::lastIndexOf($filename, '.') + 1));
		}*/
		
		// get image data
		if (($imageData = @getImageSize($tmpName)) === false) {
			throw new UserInputException('upload', 'badImage');
		}
		
		// get file extension by mime
		$fileExtension = ImageUtil::getExtensionByMimeType($imageData['mime']);
		
		// check file extension
		if (!in_array($fileExtension, self::getAllowedFileExtensions())) {
			throw new UserInputException('upload', 'illegalExtension');
		}
		
		// get avatar size
		$width = $imageData[0];
		$height = $imageData[1];
		if (!$width || !$height) {
			throw new UserInputException('upload', 'badImage');
		}
		
		// get filesize
		$filesize = $usedQuota = intval(@filesize($tmpName));
		
		// check size
		if ($width > WCF::getUser()->getPermission('user.gallery.maxWidth') || $height > WCF::getUser()->getPermission('user.gallery.maxHeight') || $filesize > WCF::getUser()->getPermission('user.gallery.maxSize')) {
			throw new UserInputException('upload', 'tooLarge');
		}
		
		// get exif data
		$maker = $camera = '';
		$creationTime = $latitude = $longitude = 0;
		if (function_exists('exif_read_data')) {
			$exifData = @exif_read_data($tmpName);
			if (!empty($exifData['Make'])) {
				$maker = $exifData['Make'];
			}
			if (!empty($exifData['Model'])) {
				$camera = $exifData['Model'];
				if ($maker != '' && strpos($camera, $maker) === false) {
					$camera = $maker.' '.$camera;
				}
			}
			if (!empty($exifData['DateTime'])) {
				$creationTime = @intval(strtotime($exifData['DateTime']));
			}
			
			// location
			if (isset($exifData['GPSLatitudeRef']) && isset($exifData['GPSLatitude']) && isset($exifData['GPSLongitudeRef']) && isset($exifData['GPSLongitude'])) {
				// latitude
				$degrees = (isset($exifData['GPSLatitude'][0]) ? self::convertCoordinateToDecimal($exifData['GPSLatitude'][0]) : 0.0);
				$minutes = (isset($exifData['GPSLatitude'][1]) ? self::convertCoordinateToDecimal($exifData['GPSLatitude'][1]) : 0.0);
				$seconds = (isset($exifData['GPSLatitude'][2]) ? self::convertCoordinateToDecimal($exifData['GPSLatitude'][2]) : 0.0);
				$latitude = ($degrees * 60.0 + (($minutes * 60.0 + $seconds) / 60.0)) / 60.0;
				if ($exifData['GPSLatitudeRef'] == 'S') $latitude *= -1;
				
				// longitude
				$degrees = (isset($exifData['GPSLongitude'][0]) ? self::convertCoordinateToDecimal($exifData['GPSLongitude'][0]) : 0.0);
				$minutes = (isset($exifData['GPSLongitude'][1]) ? self::convertCoordinateToDecimal($exifData['GPSLongitude'][1]) : 0.0);
				$seconds = (isset($exifData['GPSLongitude'][2]) ? self::convertCoordinateToDecimal($exifData['GPSLongitude'][2]) : 0.0);
				$longitude = ($degrees * 60.0 + (($minutes * 60.0 + $seconds) / 60.0)) / 60.0;
				if ($exifData['GPSLongitudeRef'] == 'W') $longitude *= -1;
			}
		}
		
		// use filename as title
		if (empty($title)) {
			$title = $filename;
			// slice file extension
			$position = StringUtil::lastIndexOf($title, '.');
			if ($position !== false && $position !== 0) {
				$title = StringUtil::substring($title, 0, $position);
			}
		}
		
		// create hash
		$photoHash = StringUtil::substring(StringUtil::getRandomID(), 0, 8);
		
		// save image
		$sql = "INSERT INTO	wcf".WCF_N."_user_gallery
					(ownerID, albumID, title, description, filename, fileExtension, photoHash, filesize, usedQuota, uploadTime, width, height, creationTime, camera, latitude, longitude)
			VALUES		(".$ownerID.", ".$albumID.", '".escapeString($title)."', '".escapeString($description)."', '".escapeString($filename)."', '".escapeString($fileExtension)."', '".$photoHash."', ".$filesize.", ".$usedQuota.", ".TIME_NOW.", ".$width.", ".$height.", ".$creationTime.", '".escapeString($camera)."', ".$latitude.", ".$longitude.")";
		WCF::getDB()->sendQuery($sql);
		
		// get id
		$photoID = WCF::getDB()->getInsertID("wcf".WCF_N."_user_gallery", 'photoID');
		
		// copy file
		if (!@copy($tmpName, WCF_DIR.'images/photos/photo-'.$photoID.'-'.$photoHash.'.'.$fileExtension)) {
			// copy failed, delete photo
			@unlink($tmpName);
			$sql = "DELETE FROM	wcf".WCF_N."_user_gallery
				WHERE		photoID = ".$photoID;
			WCF::getDB()->sendQuery($sql);
			throw new UserInputException('upload', 'copyFailed');
		}
		// set permissions
		@chmod(WCF_DIR.'images/photos/photo-'.$photoID.'-'.$photoHash.'.'.$fileExtension, 0666);
		
		// update album
		if ($albumID > 0) {
			$sql = "UPDATE	wcf".WCF_N."_user_gallery_album
				SET	photos = photos + 1,
					lastUpdateTime = ".TIME_NOW.",
					coverPhotoID = CASE WHEN coverPhotoID = 0 THEN ".$photoID." ELSE coverPhotoID END
				WHERE	albumID = ".$albumID;
			WCF::getDB()->sendQuery($sql);
		}
		
		// get new object
		$photo = new UserGalleryPhotoEditor($photoID);
		// create thumbails
		$photo->createThumbnails();
		
		return $photo;
	}
	
	/**
	 * Updates this photo.
	 * 
	 * @param	string		$title
	 * @param	string		$description
	 * @param	integer		$albumID
	 */
	public function update($title, $description = '', $albumID = 0) {
		// update data
		$sql = "UPDATE	wcf".WCF_N."_user_gallery
			SET	title = '".escapeString($title)."',
				description = '".escapeString($description)."',
				albumID = ".$albumID."
			WHERE	photoID = ".$this->photoID;
		WCF::getDB()->sendQuery($sql);
		
		// update album if necessary
		if ($albumID != $this->albumID) {
			// update old album
			if ($this->albumID != 0) {
				$sql = "UPDATE	wcf".WCF_N."_user_gallery_album user_gallery_album
					SET	photos = photos - 1,
						coverPhotoID = CASE WHEN coverPhotoID = ".$this->photoID." THEN IFNULL((SELECT photoID FROM wcf".WCF_N."_user_gallery WHERE albumID = user_gallery_album.albumID ORDER BY uploadTime LIMIT 1), 0) ELSE coverPhotoID END
					WHERE	albumID = ".$this->albumID;
				WCF::getDB()->sendQuery($sql);
			}
			
			// update new album
			if ($albumID != 0) {
				$sql = "UPDATE	wcf".WCF_N."_user_gallery_album
					SET	photos = photos + 1,
						lastUpdateTime = ".TIME_NOW.",
						coverPhotoID = CASE WHEN coverPhotoID = 0 THEN ".$this->photoID." ELSE coverPhotoID END
					WHERE	albumID = ".$albumID;
				WCF::getDB()->sendQuery($sql);
			}
		}
	}
	
	/**
	 * Deletes this photo.
	 */
	public function delete() {
		// delete comments
		$sql = "DELETE FROM	wcf".WCF_N."_user_gallery_comment
			WHERE		photoID = ".$this->photoID;
		WCF::getDB()->sendQuery($sql);
		
		// delete photo
		$sql = "DELETE FROM	wcf".WCF_N."_user_gallery
			WHERE		photoID = ".$this->photoID;
		WCF::getDB()->sendQuery($sql);
		
		if ($this->albumID != 0) {
			// update album
			$sql = "UPDATE	wcf".WCF_N."_user_gallery_album user_gallery_album
				SET	photos = photos - 1,
					coverPhotoID = CASE WHEN coverPhotoID = ".$this->photoID." THEN IFNULL((SELECT photoID FROM wcf".WCF_N."_user_gallery WHERE albumID = user_gallery_album.albumID ORDER BY uploadTime LIMIT 1), 0) ELSE coverPhotoID END
				WHERE	albumID = ".$this->albumID;
			WCF::getDB()->sendQuery($sql);
		}
		
		// delete files
		@unlink(WCF_DIR.'images/photos/photo-'.$this->photoID.($this->photoHash ? '-'.$this->photoHash : '').'.'.$this->fileExtension);
		if ($this->hasQuadraticThumbnail == 1) @unlink(WCF_DIR.'images/photos/thumbnails/quadratic/photo-'.$this->photoID.($this->photoHash ? '-'.$this->photoHash : '').'.'.$this->fileExtension);
		if ($this->hasTinyThumbnail == 1) @unlink(WCF_DIR.'images/photos/thumbnails/tiny/photo-'.$this->photoID.($this->photoHash ? '-'.$this->photoHash : '').'.'.$this->fileExtension);
		if ($this->hasSmallThumbnail == 1) @unlink(WCF_DIR.'images/photos/thumbnails/small/photo-'.$this->photoID.($this->photoHash ? '-'.$this->photoHash : '').'.'.$this->fileExtension);
		if ($this->hasMediumThumbnail == 1) @unlink(WCF_DIR.'images/photos/thumbnails/medium/photo-'.$this->photoID.($this->photoHash ? '-'.$this->photoHash : '').'.'.$this->fileExtension);
		if ($this->hasLargeThumbnail == 1) @unlink(WCF_DIR.'images/photos/thumbnails/large/photo-'.$this->photoID.($this->photoHash ? '-'.$this->photoHash : '').'.'.$this->fileExtension);
		
		// delete tags
		if (MODULE_TAGGING) {
			require_once(WCF_DIR.'lib/data/tag/TagEngine.class.php');
			$taggable = TagEngine::getInstance()->getTaggable('com.woltlab.wcf.user.gallery.photo');
			
			$sql = "DELETE FROM	wcf".WCF_N."_tag_to_object
				WHERE 		taggableID = ".$taggable->getTaggableID()."
						AND objectID = ".$this->photoID;
			WCF::getDB()->registerShutdownUpdate($sql);
		}
	}
	
	/**
	 * Creates thumbnails of this photo.
	 */
	public function createThumbnails() {
		$thumbnailSize = 0;
		// quadratic 75x75
		$quadratic = 0;
		if ($this->width >= 75 && $this->height >= 75) {
			$targetFile = WCF_DIR.'images/photos/thumbnails/quadratic/photo-'.$this->photoID.($this->photoHash ? '-'.$this->photoHash : '').'.'.$this->fileExtension;
			$thumbnail = new Thumbnail(WCF_DIR.'images/photos/photo-'.$this->photoID.($this->photoHash ? '-'.$this->photoHash : '').'.'.$this->fileExtension, 75, 75, false, null, false, true);
		
			// get thumbnail
			try {
				if (($thumbnailData = $thumbnail->makeThumbnail())) {
					// save thumbnail
					$file = new File($targetFile);
					$file->write($thumbnailData);
					unset($thumbnailData);
					$file->close();
					$quadratic = 1;
					$thumbnailSize += @filesize($targetFile);
					
					// set permissions
					@chmod($targetFile, 0666);
				}
			}
			catch (Exception $e) {}
		}
		
		// tiny 150x150
		$tiny = 0;
		if ($this->width > 150 || $this->height > 150) {
			$targetFile = WCF_DIR.'images/photos/thumbnails/tiny/photo-'.$this->photoID.($this->photoHash ? '-'.$this->photoHash : '').'.'.$this->fileExtension;
			$thumbnail = new Thumbnail(WCF_DIR.'images/photos/photo-'.$this->photoID.($this->photoHash ? '-'.$this->photoHash : '').'.'.$this->fileExtension, 150, 150, false, null, false);
		
			// get thumbnail
			try {
				if (($thumbnailData = $thumbnail->makeThumbnail())) {
					// save thumbnail
					$file = new File($targetFile);
					$file->write($thumbnailData);
					unset($thumbnailData);
					$file->close();
					$tiny = 1;
					$thumbnailSize += @filesize($targetFile);
					
					// set permissions
					@chmod($targetFile, 0666);
				}
			}
			catch (Exception $e) {}
		}
		
		// small 240x240
		$small = 0;
		if ($this->width > 240 || $this->height > 240) {
			$targetFile = WCF_DIR.'images/photos/thumbnails/small/photo-'.$this->photoID.($this->photoHash ? '-'.$this->photoHash : '').'.'.$this->fileExtension;
			$thumbnail = new Thumbnail(WCF_DIR.'images/photos/photo-'.$this->photoID.($this->photoHash ? '-'.$this->photoHash : '').'.'.$this->fileExtension, 240, 240, false, null, false);
		
			// get thumbnail
			try {
				if (($thumbnailData = $thumbnail->makeThumbnail())) {
					// save thumbnail
					$file = new File($targetFile);
					$file->write($thumbnailData);
					unset($thumbnailData);
					$file->close();
					$small = 1;
					$thumbnailSize += @filesize($targetFile);
					
					// set permissions
					@chmod($targetFile, 0666);
				}
			}
			catch (Exception $e) {}
		}
		
		// medium 500x500
		$medium = 0;
		if ($this->width > 500 || $this->height > 500) {
			$targetFile = WCF_DIR.'images/photos/thumbnails/medium/photo-'.$this->photoID.($this->photoHash ? '-'.$this->photoHash : '').'.'.$this->fileExtension;
			$thumbnail = new Thumbnail(WCF_DIR.'images/photos/photo-'.$this->photoID.($this->photoHash ? '-'.$this->photoHash : '').'.'.$this->fileExtension, 500, 500, false, null, false);
		
			// get thumbnail
			try {
				if (($thumbnailData = $thumbnail->makeThumbnail())) {
					// save thumbnail
					$file = new File($targetFile);
					$file->write($thumbnailData);
					unset($thumbnailData);
					$file->close();
					$medium = 1;
					$thumbnailSize += @filesize($targetFile);
					
					// set permissions
					@chmod($targetFile, 0666);
				}
			}
			catch (Exception $e) {}
		}

		// large 1024x1024
		$large = 0;
		if ($this->width > 1024 || $this->height > 1024) {
			$targetFile = WCF_DIR.'images/photos/thumbnails/large/photo-'.$this->photoID.($this->photoHash ? '-'.$this->photoHash : '').'.'.$this->fileExtension;
			$thumbnail = new Thumbnail(WCF_DIR.'images/photos/photo-'.$this->photoID.($this->photoHash ? '-'.$this->photoHash : '').'.'.$this->fileExtension, 1024, 1024, false, null, false);
		
			// get thumbnail
			try {
				if (($thumbnailData = $thumbnail->makeThumbnail())) {
					// save thumbnail
					$file = new File($targetFile);
					$file->write($thumbnailData);
					unset($thumbnailData);
					$file->close();
					$large = 1;
					$thumbnailSize += @filesize($targetFile);
					
					// set permissions
					@chmod($targetFile, 0666);
				}
			}
			catch (Exception $e) {}
		}
		
		// update database
		$sql = "UPDATE	wcf".WCF_N."_user_gallery
			SET	hasQuadraticThumbnail = ".$quadratic.",
				hasTinyThumbnail = ".$tiny.",
				hasSmallThumbnail = ".$small.",
				hasMediumThumbnail = ".$medium.",
				hasLargeThumbnail = ".$large.",
				usedQuota = filesize + ".$thumbnailSize."
			WHERE	photoID = ".$this->photoID;
		WCF::getDB()->sendQuery($sql);
		
		// update object data
		$this->data['hasQuadraticThumbnail'] = $quadratic;
		$this->data['hasTinyThumbnail'] = $tiny;
		$this->data['hasSmallThumbnail'] = $small;
		$this->data['hasMediumThumbnail'] = $medium;
		$this->data['hasLargeThumbnail'] = $large;
		$this->data['usedQuota'] = $this->filesize + $thumbnailSize;
	}
	
	/**
	 * Updates the tags of this photo.
	 * 
	 * @param	array<string>		$tagArray
	 */
	public function updateTags($tagArray) {
		// include files
		require_once(WCF_DIR.'lib/data/tag/TagEngine.class.php');
		require_once(WCF_DIR.'lib/data/user/gallery/TaggedUserGalleryPhoto.class.php');
		
		// save tags
		$tagged = new TaggedUserGalleryPhoto(null, array(
			'photoID' => $this->photoID,
			'taggable' => TagEngine::getInstance()->getTaggable('com.woltlab.wcf.user.gallery.photo')
		));

		$languageID = 0;
		if (count(Language::getAvailableContentLanguages()) > 0) {
			$languageID = WCF::getLanguage()->getLanguageID();
		}
		
		// delete old tags
		TagEngine::getInstance()->deleteObjectTags($tagged, array($languageID));
		
		// save new tags
		if (count($tagArray) > 0) {
			TagEngine::getInstance()->addTags($tagArray, $tagged, $languageID);
		}
	}
	
	/**
	 * Converts the format of exif geo tagging coordinates.
	 */
	private static function convertCoordinateToDecimal($coordinate) {
		$result = 0.0;
		$coordinateData = explode('/', $coordinate);
		for ($i = 0, $j = count($coordinateData); $i < $j; $i++) {
			if ($coordinateData[$i] != 0.0) {
				if ($i == 0) $result = (float) $coordinateData[0];
				else $result /= (float) $coordinateData[$i];
			}
		}
		
		return $result;
	}
	
	/**
	 * Returns a list of allowed file extensions.
	 * 
	 * @return	array<string>
	 */
	public static function getAllowedFileExtensions() {
		if (self::$allowedFileExtensions === null) {
			self::$allowedFileExtensions = array();
			self::$allowedFileExtensions = array_unique(explode("\n", StringUtil::unifyNewlines(WCF::getUser()->getPermission('user.gallery.allowedFileExtensions'))));
			self::$allowedFileExtensions = array_diff(self::$allowedFileExtensions, self::$illegalFileExtensions);
		}
		
		return self::$allowedFileExtensions;
	}
}
?>