<?php
// wcf imports
require_once(WCF_DIR.'lib/form/AbstractForm.class.php');
require_once(WCF_DIR.'lib/data/user/UserProfileFrame.class.php');
require_once(WCF_DIR.'lib/data/user/gallery/UserGalleryPhotoEditor.class.php');
require_once(WCF_DIR.'lib/data/user/gallery/album/UserGalleryAlbum.class.php');

/**
 * Shows the form for adding gallery photos.
 * 
 * @author 	Marcel Werk
 * @copyright	2001-2009 WoltLab GmbH
 * @license	WoltLab Burning Board License <http://www.woltlab.com/products/burning_board/license.php>
 * @package	com.woltlab.wcf.user.gallery
 * @subpackage	form
 * @category 	Community Framework (commercial)
 */
class UserGalleryPhotoAddForm extends AbstractForm {
	// system
	public $templateName = 'userGalleryPhotoAdd';
	
	/**
	 * number of possible uploads
	 * 
	 * @var	integer
	 */
	public $freePhotos = 0;
	
	/**
	 * free quota
	 * 
	 * @var	integer
	 */
	public $freeQuota = 0;
	
	/**
	 * list of successfully uploaded photos
	 * 
	 * @var	array<UserGalleryPhotoEditor>
	 */
	public $photos = array();
	
	// form parameters
	public $title = '';
	public $description = '';
	public $albumID = 0;
	public $upload = null;
	public $tags = '';
	
	/**
	 * user profile frame
	 * 
	 * @var UserProfileFrame
	 */
	public $frame = null;
	
	/**
	 * @see	Page::readParameters()
	 */
	public function readParameters() {
		parent::readParameters();
		
		// get profile frame
		$this->frame = new UserProfileFrame($this, WCF::getUser()->userID);
		
		// check permissions
		WCF::getUser()->checkPermission('user.gallery.canUseGallery');
		
		// get album id
		if (isset($_REQUEST['albumID'])) $this->albumID = intval($_REQUEST['albumID']);
		
		// get number of possible uploads
		$this->freePhotos = WCF::getUser()->getPermission('user.gallery.maxPhotos') - UserGalleryPhoto::countUserPhotos();
		if ($this->freePhotos <= 0) {
			throw new NamedUserException(WCF::getLanguage()->get('wcf.user.gallery.photo.error.tooManyPhotos'));
		}
		// get free quota
		$this->freeQuota = WCF::getUser()->getPermission('user.gallery.quota') * 1000000 - UserGalleryPhoto::getUsedQuota();
		if ($this->freeQuota <= 0) {
			throw new NamedUserException(WCF::getLanguage()->get('wcf.user.gallery.photo.error.quotaFull'));
		}
	}
	
	/**
	 * @see	Form::readFormParameters()
	 */
	public function readFormParameters() {
		parent::readFormParameters();
		
		if (isset($_POST['title'])) $this->title = StringUtil::trim($_POST['title']);
		if (isset($_POST['description'])) $this->description = StringUtil::trim($_POST['description']);
		if (isset($_POST['tags'])) $this->tags = StringUtil::trim($_POST['tags']);
		if (isset($_FILES['upload'])) $this->upload = $_FILES['upload'];
	}
	
	/**
	 * @see	Form::validate()
	 */
	public function validate() {
		parent::validate();
		
		// album id
		$this->validateAlbum();
		
		// upload
		$this->validateUpload();
	}
	
	/**
	 * Validates the selected album.
	 */
	protected function validateAlbum() {
		if ($this->albumID != 0) {
			$album = new UserGalleryAlbum($this->albumID);
			if (!$album->albumID || $album->ownerID != $this->frame->getUserID()) {
				$this->albumID = 0;
			}
		}
	}
	
	/**
	 * Validates the uploaded files.
	 */
	protected function validateUpload() {
		if (isset($this->upload['name']) && count($this->upload['name'])) {
			$errors = array();
			for ($x = 0, $y = count($this->upload['name']); $x < $y; $x++) {
				if (!empty($this->upload['name'][$x])) {
					try {
						// check quota
						if ($this->freePhotos <= 0) {
							throw new UserInputException('upload', 'tooManyPhotos');
						}
						if ($this->freeQuota - @filesize($this->upload['tmp_name'][$x]) <= 0) {
							throw new UserInputException('upload', 'quotaFull');
						}
						
						// save photo
						$this->photos[] = UserGalleryPhotoEditor::create(WCF::getUser()->userID, $this->upload['tmp_name'][$x], $this->upload['name'][$x], $this->title, $this->description, $this->albumID);
						
						// update quota
						$this->freePhotos--;
						$this->freeQuota -= @filesize($this->upload['tmp_name'][$x]);
					}
					catch (UserInputException $e) {
						$errors[] = array('errorType' => $e->getType(), 'filename' => $this->upload['name'][$x]);
					}
				}
			}
			
			// save tags
			if (MODULE_TAGGING) {
				$tagArray = TaggingUtil::splitString($this->tags);
				if (count($tagArray)) {
					foreach ($this->photos as $photo) {
						$photo->updateTags($tagArray);
					}
				}
			}
			
			// show success message
			WCF::getTPL()->assign('photos', $this->photos);
			if (count($this->photos) > 0) WCF::getTPL()->assign('success', true);
			
			// show error message
			if (count($errors)) {
				throw new UserInputException('upload', $errors);
			}
			
		}
		else {
			throw new UserInputException('upload');
		}
	}
	
	/**
	 * @see	Form::save()
	 */
	public function save() {
		parent::save();
		$this->saved();
		
		// reset values
		$this->title = $this->description = $this->tags = '';
	}
	
	/**
	 * @see Page::assignVariables()
	 */
	public function assignVariables() {
		parent::assignVariables();
		
		$this->frame->assignVariables();
		WCF::getTPL()->assign(array(
			'title' => $this->title,
			'description' => $this->description,
			'albumID' => $this->albumID,
			'action' => 'add',
			'availableAlbums' => UserGalleryAlbum::getAlbums($this->frame->getUserID()),
			'tags' => $this->tags,
			'allowedExtensions' => implode(", ", array_unique(explode("\n", StringUtil::unifyNewlines(WCF::getUser()->getPermission('user.gallery.allowedFileExtensions'))))),
			'freePhotos' => $this->freePhotos,
			'freeQuota' => $this->freeQuota 
		));
	}
	
	/**
	 * @see Page::show()
	 */
	public function show() {
		// set active menu item
		UserProfileMenu::getInstance()->setActiveMenuItem('wcf.user.profile.menu.link.gallery');
		
		// check permission
		WCF::getUser()->checkPermission('user.gallery.canUseGallery');
		
		if (!MODULE_USER_GALLERY) {
			throw new IllegalLinkException();
		}
		
		parent::show();
	}
}
?>