<?php
// wcf imports
require_once(WCF_DIR.'lib/data/DatabaseObject.class.php');

/**
 * Represents a gallery photo.
 *
 * @author 	Marcel Werk
 * @copyright	2001-2009 WoltLab GmbH
 * @license	WoltLab Burning Board License <http://www.woltlab.com/products/burning_board/license.php>
 * @package	com.woltlab.wcf.user.gallery
 * @subpackage	data.user.gallery
 * @category 	Community Framework (commercial)
 */
class UserGalleryPhoto extends DatabaseObject {
	/**
	 * Creates a new UserGalleryPhoto object.
	 *
	 * @param	integer		$photoID
	 * @param 	array<mixed>	$row
	 */
	public function __construct($photoID, $row = null) {
		if ($photoID !== null) {
			$sql = "SELECT	*
				FROM 	wcf".WCF_N."_user_gallery
				WHERE 	photoID = ".$photoID;
			$row = WCF::getDB()->getFirstRow($sql);
		}
		parent::__construct($row);
	}
	
	/**
	 * Returns the title of this photo.
	 * 
	 * @return	string
	 */
	public function __toString() {
		return $this->title;
	}
	
	/**
	 * Returns the path to this photo.
	 * 
	 * @param	string		$size
	 * @return	string
	 */
	public function getPhoto($size = '') {
		switch ($size) {
			case 'quadratic': 
				if ($this->hasQuadraticThumbnail == 1) {
					return RELATIVE_WCF_DIR.'images/photos/thumbnails/quadratic/photo-'.$this->photoID.($this->photoHash ? '-'.$this->photoHash : '').'.'.$this->fileExtension;
				}
				else if ($this->width <= 75 && $this->height <= 75) {
					return RELATIVE_WCF_DIR.'images/photos/photo-'.$this->photoID.($this->photoHash ? '-'.$this->photoHash : '').'.'.$this->fileExtension;
				}
				break;
			case 'tiny':
				if ($this->hasTinyThumbnail == 1) {
					return RELATIVE_WCF_DIR.'images/photos/thumbnails/tiny/photo-'.$this->photoID.($this->photoHash ? '-'.$this->photoHash : '').'.'.$this->fileExtension;
				}
				else if ($this->width <= 150 && $this->height <= 150) {
					return RELATIVE_WCF_DIR.'images/photos/photo-'.$this->photoID.($this->photoHash ? '-'.$this->photoHash : '').'.'.$this->fileExtension;
				}
				break;
			case 'small': 
				if ($this->hasSmallThumbnail == 1) {
					return RELATIVE_WCF_DIR.'images/photos/thumbnails/small/photo-'.$this->photoID.($this->photoHash ? '-'.$this->photoHash : '').'.'.$this->fileExtension;
				}
				else if ($this->width <= 240 && $this->height <= 240) {
					return RELATIVE_WCF_DIR.'images/photos/photo-'.$this->photoID.($this->photoHash ? '-'.$this->photoHash : '').'.'.$this->fileExtension;
				}
				break;
			case 'medium':
				if ($this->hasMediumThumbnail == 1) {
					return RELATIVE_WCF_DIR.'images/photos/thumbnails/medium/photo-'.$this->photoID.($this->photoHash ? '-'.$this->photoHash : '').'.'.$this->fileExtension;
				}
				else if ($this->width <= 500 && $this->height <= 500) {
					return RELATIVE_WCF_DIR.'images/photos/photo-'.$this->photoID.($this->photoHash ? '-'.$this->photoHash : '').'.'.$this->fileExtension;
				}
				break;
			case 'large': 
				if ($this->hasLargeThumbnail == 1) {
					return RELATIVE_WCF_DIR.'images/photos/thumbnails/large/photo-'.$this->photoID.($this->photoHash ? '-'.$this->photoHash : '').'.'.$this->fileExtension;
				}
				else if ($this->width <= 1024 && $this->height <= 1024) {
					return RELATIVE_WCF_DIR.'images/photos/photo-'.$this->photoID.($this->photoHash ? '-'.$this->photoHash : '').'.'.$this->fileExtension;
				}
				break;
			default:
				return RELATIVE_WCF_DIR.'images/photos/photo-'.$this->photoID.($this->photoHash ? '-'.$this->photoHash : '').'.'.$this->fileExtension;
		}
		
		return self::getDefaultPhoto($size);
	}
	
	/**
	 * Returns the tags of this photo.
	 * 
	 * @return	array<Tag>
	 */
	public function getTags($languageIDArray) {
		// include files
		require_once(WCF_DIR.'lib/data/tag/TagEngine.class.php');
		require_once(WCF_DIR.'lib/data/user/gallery/TaggedUserGalleryPhoto.class.php');
		
		// get tags
		return TagEngine::getInstance()->getTagsByTaggedObject(new TaggedUserGalleryPhoto(null, array(
			'photoID' => $this->photoID,
			'taggable' => TagEngine::getInstance()->getTaggable('com.woltlab.wcf.user.gallery.photo')
		)), $languageIDArray);
	}
	
	/**
	 * Returns the formatted description of this photo.
	 * 
	 * @return	string
	 */
	public function getFormattedDescription() {
		if ($this->description) {
			return nl2br(StringUtil::encodeHTML($this->description));
		}
		
		return '';
	}
	
	/**
	 * Returns the number of views per day.
	 *
	 * @return	float
	 */
	public function getViewsPerDay() {
		$age = round(((TIME_NOW - $this->uploadTime) / 86400), 0);
		if ($age > 0) {
			return $this->views / $age;
		}
		
		return $this->views;
	}
	
	/**
	 * Returns the width of this photo.
	 * 
	 * @param	string		$size
	 * @return	integer
	 */
	public function getPhotoWidth($size = '') {
		switch ($size) {
			case 'quadratic': 
				return $this->getQuadraticWidth();
				break;
			case 'tiny':
				return $this->getTinyWidth();
				break;
			case 'small': 
				return $this->getSmallWidth();
				break;
			case 'medium':
				return $this->getMediumWidth();
				break;
			case 'large': 
				return $this->getLargeWidth();
				break;
			default:
				return $this->width;
		}
	}
	
	/**
	 * Returns the height of this photo.
	 * 
	 * @param	string		$size
	 * @return	integer
	 */
	public function getPhotoHeight($size = '') {
		switch ($size) {
			case 'quadratic': 
				return $this->getQuadraticHeight();
				break;
			case 'tiny':
				return $this->getTinyHeight();
				break;
			case 'small': 
				return $this->getSmallHeight();
				break;
			case 'medium':
				return $this->getMediumHeight();
				break;
			case 'large': 
				return $this->getLargeHeight();
				break;
			default:
				return $this->height;
		}
	}
	
	/**
	 * Returns the width of a thumbnail size.
	 * 
	 * @param	integer		$max
	 * @return	integer
	 */
	protected function getThumbnailWidth($max) {
		if ($this->width > $this->height) {
			return $max;
		}
		else {
			return round($this->width * ($max / $this->height), 0);
		}
	}
	
	/**
	 * Returns the height of a thumbnail size.
	 * 
	 * @param	integer		$max
	 * @return	integer
	 */
	protected function getThumbnailHeight($max) {
		if ($this->height > $this->width) {
			return $max;
		}
		else {
			return round($this->height * ($max / $this->width), 0);
		}
	}
	
	/**
	 * Returns the width of the tiny thumbnail.
	 * 
	 * @return	integer
	 */
	public function getQuadraticWidth() {
		if ($this->hasQuadraticThumbnail != 1 && $this->width <= 75) return $this->width;
		return 75;
	}
	
	/**
	 * Returns the height of the tiny thumbnail.
	 * 
	 * @return	integer
	 */
	public function getQuadraticHeight() {
		if ($this->hasQuadraticThumbnail != 1 && $this->height <= 75) return $this->height;
		return 75;
	}
	
	/**
	 * Returns the width of the tiny thumbnail.
	 * 
	 * @return	integer
	 */
	public function getTinyWidth() {
		if ($this->hasTinyThumbnail != 1 && $this->width <= 150) return $this->width;
		return $this->getThumbnailWidth(150);
	}
	
	/**
	 * Returns the height of the tiny thumbnail.
	 * 
	 * @return	integer
	 */
	public function getTinyHeight() {
		if ($this->hasTinyThumbnail != 1 && $this->height <= 150) return $this->height;
		return $this->getThumbnailHeight(150);
	}
	
	/**
	 * Returns the width of the small thumbnail.
	 * 
	 * @return	integer
	 */
	public function getSmallWidth() {
		if ($this->hasSmallThumbnail != 1 && $this->width <= 240) return $this->width;
		return $this->getThumbnailWidth(240);
	}
	
	/**
	 * Returns the height of the small thumbnail.
	 * 
	 * @return	integer
	 */
	public function getSmallHeight() {
		if ($this->hasSmallThumbnail != 1 && $this->height <= 240) return $this->height;
		return $this->getThumbnailHeight(240);
	}
	
	/**
	 * Returns the width of the medium thumbnail.
	 * 
	 * @return	integer
	 */
	public function getMediumWidth() {
		if ($this->hasMediumThumbnail != 1 && $this->width <= 500) return $this->width;
		return $this->getThumbnailWidth(500);
	}
	
	/**
	 * Returns the height of the medium thumbnail.
	 * 
	 * @return	integer
	 */
	public function getMediumHeight() {
		if ($this->hasMediumThumbnail != 1 && $this->height <= 500) return $this->height;
		return $this->getThumbnailHeight(500);
	}
	
	/**
	 * Returns the width of the large thumbnail.
	 * 
	 * @return	integer
	 */
	public function getLargeWidth() {
		if ($this->hasLargeThumbnail != 1 && $this->width <= 1024) return $this->width;
		return $this->getThumbnailWidth(1024);
	}
	
	/**
	 * Returns the height of the large thumbnail.
	 * 
	 * @return	integer
	 */
	public function getLargeHeight() {
		if ($this->hasLargeThumbnail != 1 && $this->height <= 1024) return $this->height;
		return $this->getThumbnailHeight(1024);
	}
	
	/**
	 * Returns the formatted coordinates of this photo.
	 * 
	 * @return	string
	 */
	public function getCoordinates() {
		if ($this->latitude && $this->longitude) {
			// latitude
			if ($this->latitude < 0) $latitudeRef = 'S';
			else $latitudeRef = 'N';
			$latitude = abs($this->latitude);
			// degrees
			$latDegrees = floor($latitude);
			$latitude -= $latDegrees;
			// minutes
			$latitude *= 60.0;
			$latMinutes = floor($latitude);
			$latitude -= $latMinutes;
			// seconds
			$latitude *= 60.0;
			$latSeconds = round($latitude, 0);
			
			// longitude
			if ($this->longitude < 0) $longitudeRef = 'W';
			else $longitudeRef = 'E';
			$longitude = abs($this->longitude);
			// degrees
			$longDegrees = floor($longitude);
			$longitude -= $longDegrees;
			// minutes
			$longitude *= 60.0;
			$longMinutes = floor($longitude);
			$longitude -= $longMinutes;
			// seconds
			$longitude *= 60.0;
			$longSeconds = round($longitude, 0);
			
			// return result
			return WCF::getLanguage()->getDynamicVariable('wcf.user.gallery.photo.coordinate.value', array(
				'latDegrees' => $latDegrees,
				'latMinutes' => $latMinutes,
				'latSeconds' => $latSeconds,
				'latitudeRef' => $latitudeRef,
				'longDegrees' => $longDegrees,
				'longMinutes' => $longMinutes,
				'longSeconds' => $longSeconds,
				'longitudeRef' => $longitudeRef
			));
		}
		
		return '';
	}
	
	/**
	 * Returns true, if the active user can comment this photo.
	 * 
	 * @return	boolean
	 */
	public function isCommentable() {
		return WCF::getUser()->getPermission('user.gallery.canComment');
	}
	
	/**
	 * Returns true, if the active user can edit this photo.
	 * 
	 * @return	boolean
	 */
	public function isEditable() {
		if (($this->ownerID == WCF::getUser()->userID && WCF::getUser()->getPermission('user.gallery.canUseGallery')) || WCF::getUser()->getPermission('mod.gallery.canEditPhoto')) {
			return true;
		}
		return false;
	}
	
	/**
	 * Returns true, if the active user can delete this photo.
	 * 
	 * @return	boolean
	 */
	public function isDeletable() {
		if (($this->ownerID == WCF::getUser()->userID && WCF::getUser()->getPermission('user.gallery.canUseGallery')) || WCF::getUser()->getPermission('mod.gallery.canDeletePhoto')) {
			return true;
		}
		return false;
	}
	
	/**
	 * Returns the aspect ratio of this photo.
	 * 
	 * @return	string
	 */
	public function getAspectRatio() {
		if ($this->width > $this->height) {
			$ratio = $this->height / $this->width;
			$factor = 1.0 - $ratio;
			if ($ratio != 0.0 && $factor != 0.0) {
				$widthRatio = round(1 / min($ratio, $factor), 0);
				$heightRatio = round($ratio / min($ratio, $factor), 0);
				return $widthRatio.':'.$heightRatio;
			}
		}
		else {
			$ratio = $this->width / $this->height;
			$factor = 1.0 - $ratio;
			if ($ratio != 0.0 && $factor != 0.0) {
				$widthRatio = round($ratio / min($ratio, $factor), 0);
				$heightRatio = round(1 / min($ratio, $factor), 0); 
				return $widthRatio.':'.$heightRatio;
			}
		}
		
		return '1:1';
	}
	
	/**
	 * Counts the photos of a user.
	 * 
	 * @param	integer		$userID
	 * @return	integer
	 */
	public static function countUserPhotos($userID = null) {
		if ($userID === null) $userID = WCF::getUser()->userID;
		
		$sql = "SELECT	COUNT(*) AS photos
			FROM	wcf".WCF_N."_user_gallery
			WHERE	ownerID = ".$userID;
		$row = WCF::getDB()->getFirstRow($sql);
		return $row['photos'];
	}
	
	/**
	 * Counts the used quota.
	 * 
	 * @param	integer		$userID
	 * @return	integer
	 */
	public static function getUsedQuota($userID = null) {
		if ($userID === null) $userID = WCF::getUser()->userID;
		
		$sql = "SELECT	SUM(usedQuota) AS usedQuota
			FROM	wcf".WCF_N."_user_gallery
			WHERE	ownerID = ".$userID;
		$row = WCF::getDB()->getFirstRow($sql);
		if (!empty($row['usedQuota'])) return $row['usedQuota'];
		return 0;
	}
	
	/**
	 * Returns the path to the default photo.
	 * 
	 * @param	string		$size
	 * @return	string
	 */
	public static function getDefaultPhoto($size = '') {
		switch ($size) {
			case 'quadratic': 
				return RELATIVE_WCF_DIR.'images/photos/thumbnails/quadratic/default.png';
				break;
			case 'tiny':
				return RELATIVE_WCF_DIR.'images/photos/thumbnails/tiny/default.png';
				break;
			case 'small': 
				return RELATIVE_WCF_DIR.'images/photos/thumbnails/small/default.png';
				break;
			case 'medium':
				return RELATIVE_WCF_DIR.'images/photos/thumbnails/medium/default.png';
				break;
			case 'large': 
				return RELATIVE_WCF_DIR.'images/photos/thumbnails/large/default.png';
				break;
			default:
				return RELATIVE_WCF_DIR.'images/photos/default.png';
		}
	}
	
	/**
	 * Returns an empty photo object.
	 * 
	 * @return	UserGalleryPhoto
	 */
	public static function getEmptyPhotoObject() {
		return new UserGalleryPhoto(null, array(
			'fileExtension' => 'png',
			'width' => 1024,
			'height' => 768
		));
	}
}
?>